#include <algorithm>
#include <iostream>
#include <vector>

struct VisitCard {
    int spawnTime;
    int depth;
    int owner;
};

template<typename T>
class Node {
private:  // Fiedls
    T value;
    Node* parent;
    std::vector<Node*> children;
    std::vector<int> treeIndices;
public:   // Constructors
    Node();
    Node(const T&);
public:   // Methods
    T GetValue() const;
    int GetChildTreeIndex(const size_t&) const;
    void AddChild(Node&);
    void AddChild(Node&, const int&);
    void SetParent(Node&);
    void SetValue(const T&);
    Node& GetChild(const size_t&) const;
    Node& GetParent() const;
    size_t ChildCount() const;
};

template<typename T>
class Tree {
private:  // Feidls
    std::vector<Node<T>> nodes;
    std::vector<int> depths;
public:   // Constructors
    Tree() = default;
    Tree(const size_t&);
public:   // Methods
    int DepthOfNode(const size_t&) const;
    void AddNode(const T&);
    void AddNode(const Node<T>&);
    void AddChild(const size_t&, const size_t&);
    void SetParrent(const size_t&, const size_t&);
    void AssignNode(const size_t&, const T&);
    void UpdateDepths(const size_t&, const int& = 0);
};

void Untie();

int main() {
    Untie();
    int n;
    int m;
    std::cin >> n >> m;
    Tree<int> company(n);
    for (int i = 0; i < n; i++) {
        company.AssignNode(i, i + 1);
    }
    for (int i = 0; i < n - 1; i++) {
        int parent;
        std::cin >> parent;
        company.SetParrent(parent - 1, (i + 2) - 1);
        company.AddChild((i + 2) - 1, parent - 1);
    }
    std::vector<VisitCard> cards(m);
    for (auto& card : cards) {
        std::cin >> card.spawnTime >> card.owner;
    }
    company.UpdateDepths(0);
    for (auto& card : cards) {
        card.depth = company.DepthOfNode(card.owner - 1);
    }
    std::vector<int> arrivals(m);
    for (int i = 0; i < m; i++) {
        arrivals[i] = cards[i].spawnTime + cards[i].depth;
    }
    std::sort(arrivals.begin(), arrivals.end());
    int series = 1;
    int thrown = 0;
    for (int i = 1; i < m; i++) {
        if (arrivals[i] == arrivals[i - 1]) {
            series++;
        }
        else {
            thrown += series - (series % 2);
            series = 1;
        }
    }
    thrown += series - (series % 2);
    std::cout << thrown << std::endl;
    return 0;
}


void Untie() {
    std::ios_base::sync_with_stdio(false);
    std::cin.tie(nullptr);
    std::cout.tie(nullptr);
    std::cerr.tie(nullptr);
}

// Node
template<typename T>
Node<T>::Node()
: parent(nullptr) {
}
template<typename T>
Node<T>::Node(const T& value)
: value(value), parent(nullptr) {
}

// Methods
template<typename T>
T Node<T>::GetValue() const {
    return value;
}
template<typename T>
void Node<T>::AddChild(Node& node) {
    children.push_back(&node);
}
template<typename T>
void Node<T>::AddChild(Node& node, const int& index) {
    children.push_back(&node);
    treeIndices.push_back(index);
}
template<typename T>
int Node<T>::GetChildTreeIndex(const size_t& child) const {
    return treeIndices.at(child);
}
template<typename T>
void Node<T>::SetParent(Node& node) {
    parent = &node;
}
template<typename T>
void Node<T>::SetValue(const T& value) {
    this->value = value;
}
template<typename T>
Node<T>& Node<T>::GetChild(const size_t& index) const {
    return *(children.at(index));
}
template<typename T>
Node<T>& Node<T>::GetParent() const {
    return *parent;
}
template<typename T>
size_t Node<T>::ChildCount() const {
    return children.size();
}

// Tree
template<typename T>
Tree<T>::Tree(const size_t& size)
: nodes(size), depths(size) {
}

// Methods
template<typename T>
int Tree<T>::DepthOfNode(const size_t& index) const {
    return depths[index];
}
template<typename T>
void Tree<T>::AddNode(const T& value) {
    nodes.push_back(Node<T>(value));
}
template<typename T>
void Tree<T>::AddNode(const Node<T>& node) {
    nodes.push_back(node);
}
template<typename T>
void Tree<T>::SetParrent(const size_t& parent, const size_t& child) {
    nodes[child].SetParent(nodes[parent]);
}
template<typename T>
void Tree<T>::AddChild(const size_t& child, const size_t& parent) {
    nodes[parent].AddChild(nodes[child], child);
}
template<typename T>
void Tree<T>::UpdateDepths(const size_t& root, const int& depth) {
    depths[root] = depth;
    for (int i = 0; i < nodes[root].ChildCount(); i++) {
        UpdateDepths(nodes[root].GetChildTreeIndex(i), depth + 1);
    }
}
template<typename T>
void Tree<T>::AssignNode(const size_t& index, const T& value) {
    nodes[index].SetValue(value);
}