using System;
using System.Collections.Generic;

public class ConvexHullCalculator
{
    // Represents a point in a two-dimensional cartesian coordinate system.
    public struct Point
    {
        public double X;
        public double Y;

        public Point(double x, double y)
        {
            X = x;
            Y = y;
        }

        public Point(string[] s)
        {
            X = int.Parse(s[0]);
            Y = int.Parse(s[1]);
        }
    }

    private class PolarAngleComparer : IComparer<Point>
    {
        private Point referencePoint;

        public PolarAngleComparer(Point referencePoint)
        {
            this.referencePoint = referencePoint;
        }

        public int Compare(Point a, Point b)
        {
            double polarA = Math.Atan2(a.Y - referencePoint.Y, a.X - referencePoint.X);
            double polarB = Math.Atan2(b.Y - referencePoint.Y, b.X - referencePoint.X);
            return polarA.CompareTo(polarB);
        }
    }


    // Given a set of points, returns the total number of convex hulls that can be formed around the point (0,0).
    public static int CalculateConvexHulls(Point[] points)
    {
        int convexHullCount = 0;

        // Find the convex hull for each point in the set.
        for (int i = 0; i < points.Length; i++)
        {
            Point[] convexHull = FindConvexHull(points, i);
            if (convexHull.Length > 0)
            {
                convexHullCount++;
            }
        }

        return convexHullCount;
    }

    // Given a set of points and a starting point, returns the convex hull formed by the points around the starting point.
    private static Point[] FindConvexHull(Point[] points, int startIndex)
    {
        // Sort the points by polar angle with respect to the starting point.
        List<Point> list = new List<Point>(points);
        Array.Sort(points, startIndex, points.Length - startIndex, new PolarAngleComparer(points[startIndex]));

        Stack<Point> stack = new Stack<Point>();
        stack.Push(points[startIndex]);

        for (int i = startIndex + 1; i < points.Length; i++)
        {
            Point currentPoint = points[i];
            while (stack.Count >= 2)
            {
                Point lastPoint = stack.Pop();
                Point secondToLastPoint = stack.Peek();
                if (IsCounterClockwiseTurn(secondToLastPoint, lastPoint, currentPoint))
                {
                    stack.Push(lastPoint);
                    break;
                }
            }
            stack.Push(currentPoint);
        }

        return stack.ToArray();
    }

    // Given three points, returns true if the points form a counterclockwise turn, and false if they form a clockwise turn or are collinear.
    private static bool IsCounterClockwiseTurn(Point a, Point b, Point c)
    {
        double crossProduct = (b.X - a.X) * (c.Y - a.Y) - (b.Y - a.Y) * (c.X - a.X);
        return crossProduct > 0;
    }

    static void Main(string[] args)
    {
        int n = int.Parse(Console.ReadLine());
        Point[] points = new Point[n];
        for (int i = 0; i < n; i++)
            points[i] = (new Point(Console.ReadLine().Split()));

        if(n == 1)
        {
            if (points[0].X == 0 && points[0].Y == 0)
                Console.WriteLine(1);
            else
                Console.WriteLine(0);
        }

        if (n == 2)
        {
            if (points[0].X == -points[1].X && points[0].Y == -points[1].Y)
                Console.WriteLine(1);
            else
                Console.WriteLine(0);
        }

        Console.WriteLine(CalculateConvexHulls(points));
        Console.ReadLine();
    }
}