#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <stdint.h>

#define MAXN 510
#define MAXM 200000   // safe upper bound for edges in flow (will be adjusted)
#define INF64 ((long long)4e18)

typedef long long ll;

/* --- Tarjan SCC --- */
int N, M;
ll Kvals[MAXN];

int index_cnt;
int dfn[MAXN], low[MAXN], in_stack[MAXN], comp_id[MAXN], comp_cnt;
int stk[MAXN], stk_top;
vector_int {
    int *a;
    int n, cap;
};
void vec_init(vector_int *v) { v->n = 0; v->cap = 0; v->a = NULL; }
void vec_push(vector_int *v, int x) {
    if (v->n == v->cap) {
        v->cap = v->cap ? v->cap*2 : 4;
        v->a = realloc(v->a, v->cap * sizeof(int));
    }
    v->a[v->n++] = x;
}
void vec_free(vector_int *v) { free(v->a); v->a = NULL; v->n = v->cap = 0; }

vector_int g[MAXN];

void tarjan_init(int n) {
    index_cnt = 0;
    stk_top = 0;
    comp_cnt = 0;
    for (int i = 0; i < n; ++i) {
        dfn[i] = 0;
        low[i] = 0;
        in_stack[i] = 0;
        comp_id[i] = -1;
    }
}

void tarjan_dfs(int u) {
    dfn[u] = low[u] = ++index_cnt;
    stk[stk_top++] = u;
    in_stack[u] = 1;
    for (int i = 0; i < g[u].n; ++i) {
        int v = g[u].a[i];
        if (!dfn[v]) {
            tarjan_dfs(v);
            if (low[v] < low[u]) low[u] = low[v];
        } else if (in_stack[v]) {
            if (dfn[v] < low[u]) low[u] = dfn[v];
        }
    }
    if (low[u] == dfn[u]) {
        while (1) {
            int v = stk[--stk_top];
            in_stack[v] = 0;
            comp_id[v] = comp_cnt;
            if (v == u) break;
        }
        comp_cnt++;
    }
}

/* --- Dinic (long long capacities) --- */
typedef struct {
    int to;
    int rev;
    ll cap;
} Edge;

typedef struct {
    Edge *data;
    int n, cap;
} VecE;

void ve_init(VecE *v) { v->n = 0; v->cap = 0; v->data = NULL; }
void ve_push(VecE *v, Edge e) {
    if (v->n == v->cap) {
        v->cap = v->cap ? v->cap*2 : 4;
        v->data = realloc(v->data, v->cap * sizeof(Edge));
    }
    v->data[v->n++] = e;
}
void ve_free(VecE *v) { free(v->data); v->data = NULL; v->n = v->cap = 0; }

VecE adj_flow[MAXN*2]; // will allocate dynamically sized depending on comps

int Snode, Tnode, FN; // flow graph node count
int level_arr[MAXN*2];
int it_ptr[MAXN*2];

void add_flow_edge_one(int u, int v, ll c) {
    Edge e1 = {v, adj_flow[v].n, c};
    Edge e2 = {u, adj_flow[u].n, 0};
    ve_push(&adj_flow[u], e1);
    ve_push(&adj_flow[v], e2);
}

void add_flow_edge(int u, int v, ll c) {
    add_flow_edge_one(u, v, c);
}

int bfs_level() {
    for (int i = 0; i < FN; ++i) level_arr[i] = -1;
    int qh = 0, qt = 0;
    static int q[MAXN*2];
    q[qt++] = Snode;
    level_arr[Snode] = 0;
    while (qh < qt) {
        int u = q[qh++];
        for (int i = 0; i < adj_flow[u].n; ++i) {
            Edge *e = &adj_flow[u].data[i];
            if (e->cap > 0 && level_arr[e->to] < 0) {
                level_arr[e->to] = level_arr[u] + 1;
                q[qt++] = e->to;
                if (e->to == Tnode) return 1;
            }
        }
    }
    return level_arr[Tnode] >= 0;
}

ll dfs_flow(int u, ll f) {
    if (u == Tnode) return f;
    for (int *i = &it_ptr[u]; *i < adj_flow[u].n; (*i)++) {
        Edge *e = &adj_flow[u].data[*i];
        if (e->cap > 0 && level_arr[e->to] == level_arr[u] + 1) {
            ll pushed = dfs_flow(e->to, f < e->cap ? f : e->cap);
            if (pushed > 0) {
                e->cap -= pushed;
                adj_flow[e->to].data[e->rev].cap += pushed;
                return pushed;
            }
        }
    }
    return 0;
}

ll dinic_maxflow() {
    ll flow = 0;
    while (bfs_level()) {
        for (int i = 0; i < FN; ++i) it_ptr[i] = 0;
        while (1) {
            ll pushed = dfs_flow(Snode, INF64);
            if (pushed == 0) break;
            flow += pushed;
        }
    }
    return flow;
}

/* --- Helper for avoiding duplicate edges between components --- */
int comp_edge_seen[MAXN][MAXN]; // comp_cnt <= N <= 500, acceptable

int main() {
    int T;
    if (scanf("%d", &T) != 1) return 0;
    while (T--) {
        if (scanf("%d %d", &N, &M) != 2) return 0;
        for (int i = 0; i < N; ++i) {
            scanf("%lld", &Kvals[i]);
        }
        for (int i = 0; i < N; ++i) {
            vec_init(&g[i]);
        }
        for (int i = 0; i < M; ++i) {
            int a, b;
            scanf("%d %d", &a, &b);
            // nodes are 0-indexed as per statement/examples
            vec_push(&g[a], b);
        }

        // Tarjan
        tarjan_init(N);
        for (int i = 0; i < N; ++i) if (!dfn[i]) tarjan_dfs(i);

        // sum weights per component
        ll comp_weight[MAXN];
        for (int i = 0; i < comp_cnt; ++i) comp_weight[i] = 0;
        for (int i = 0; i < N; ++i) comp_weight[comp_id[i]] += Kvals[i];

        // build condensed DAG edges, avoid duplicates
        for (int i = 0; i < comp_cnt; ++i)
            for (int j = 0; j < comp_cnt; ++j)
                comp_edge_seen[i][j] = 0;

        for (int u = 0; u < N; ++u) {
            int cu = comp_id[u];
            for (int i = 0; i < g[u].n; ++i) {
                int v = g[u].a[i];
                int cv = comp_id[v];
                if (cu != cv) comp_edge_seen[cu][cv] = 1;
            }
        }

        // Build flow network:
        // nodes: 0..comp_cnt-1 correspond to components
        // source = comp_cnt, sink = comp_cnt+1
        FN = comp_cnt + 2;
        Snode = comp_cnt;
        Tnode = comp_cnt + 1;
        for (int i = 0; i < FN; ++i) ve_init(&adj_flow[i]);

        ll total_pos = 0;
        for (int i = 0; i < comp_cnt; ++i) {
            if (comp_weight[i] > 0) {
                add_flow_edge(Snode, i, comp_weight[i]);
                total_pos += comp_weight[i];
            } else if (comp_weight[i] < 0) {
                add_flow_edge(i, Tnode, -comp_weight[i]);
            }
        }
        // infinite edges for closure constraints
        for (int i = 0; i < comp_cnt; ++i) {
            for (int j = 0; j < comp_cnt; ++j) {
                if (comp_edge_seen[i][j]) {
                    add_flow_edge(i, j, INF64); // infinite capacity
                }
            }
        }

        ll mincut = dinic_maxflow();
        ll best = total_pos - mincut;
        if (best > 0) printf("%lld\n", best);
        else printf("-1\n");

        // free memory
        for (int i = 0; i < N; ++i) vec_free(&g[i]);
        for (int i = 0; i < FN; ++i) ve_free(&adj_flow[i]);
    }
    return 0;
}