#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>

typedef long long ll;

typedef struct Edge {
    int to, rev;
    ll cap;
} Edge;

typedef struct Vec {
    Edge *a;
    int sz, cap;
} Vec;

static inline void vec_init(Vec *v) {
    v->a = NULL; v->sz = 0; v->cap = 0;
}

static inline void vec_push(Vec *v, Edge e) {
    if (v->sz == v->cap) {
        v->cap = v->cap ?  v->cap * 2 :  4;
        v->a = (Edge*)realloc(v->a, v->cap * sizeof(Edge));
    }
    v->a[v->sz++] = e;
}

typedef struct Dinic {
    int n;
    Vec *g;
    int *level;
    int *it;
    int *queue;
} Dinic;

static void dinic_init(Dinic *d, int n) {
    d->n = n;
    d->g = (Vec*)malloc(n * sizeof(Vec));
    for (int i = 0; i < n; i++) vec_init(&d->g[i]);
    d->level = (int*)malloc(n * sizeof(int));
    d->it = (int*)malloc(n * sizeof(int));
    d->queue = (int*)malloc(n * sizeof(int));
}

static inline void dinic_add_edge(Dinic *d, int fr, int to, ll cap) {
    Edge fwd = {to, d->g[to].sz, cap};
    Edge rev = {fr, d->g[fr]. sz, 0};
    vec_push(&d->g[fr], fwd);
    vec_push(&d->g[to], rev);
}

static int dinic_bfs(Dinic *d, int s, int t) {
    memset(d->level, -1, d->n * sizeof(int));
    int qh = 0, qt = 0;
    d->level[s] = 0;
    d->queue[qt++] = s;
    
    while (qh < qt) {
        int v = d->queue[qh++];
        int target_level = d->level[v] + 1;
        Vec *gv = &d->g[v];
        
        for (int i = 0; i < gv->sz; i++) {
            Edge *e = &gv->a[i];
            int w = e->to;
            if (e->cap > 0 && d->level[w] < 0) {
                d->level[w] = target_level;
                d->queue[qt++] = w;
                if (w == t) return 1;
            }
        }
    }
    
    return d->level[t] >= 0;
}

static ll dinic_dfs(Dinic *d, int v, int t, ll f) {
    if (v == t) return f;
    
    Vec *gv = &d->g[v];
    int level_v = d->level[v];
    
    for (int i = d->it[v]; i < gv->sz; i++) {
        Edge *e = &gv->a[i];
        
        if (e->cap <= 0 || d->level[e->to] != level_v + 1) continue;
        
        ll pushed = dinic_dfs(d, e->to, t, f < e->cap ? f : e->cap);
        
        if (pushed > 0) {
            e->cap -= pushed;
            d->g[e->to].a[e->rev].cap += pushed;
            d->it[v] = i;
            return pushed;
        }
    }
    
    d->it[v] = gv->sz;
    return 0;
}

static ll dinic_maxflow(Dinic *d, int s, int t) {
    ll flow = 0;
    
    while (dinic_bfs(d, s, t)) {
        memset(d->it, 0, d->n * sizeof(int));
        
        while (1) {
            ll pushed = dinic_dfs(d, s, t, LLONG_MAX >> 2);
            if (pushed == 0) break;
            flow += pushed;
        }
    }
    
    return flow;
}

static void dinic_free(Dinic *d) {
    for (int i = 0; i < d->n; i++) {
        if (d->g[i].a) free(d->g[i].a);
    }
    free(d->g);
    free(d->level);
    free(d->it);
    free(d->queue);
}

int main(void) {
    int T;
    scanf("%d", &T);

    while (T--) {
        int N, M;
        scanf("%d %d", &N, &M);

        ll *K = (ll*)malloc(N * sizeof(ll));
        for (int i = 0; i < N; i++) {
            scanf("%lld", &K[i]);
        }

        int *A = (int*)malloc(M * sizeof(int));
        int *B = (int*)malloc(M * sizeof(int));
        for (int i = 0; i < M; i++) {
            scanf("%d %d", &A[i], &B[i]);
        }

        int S = N;
        int Tt = N + 1;
        Dinic din;
        dinic_init(&din, N + 2);

        ll sumPos = 0;
        
        for (int i = 0; i < N; i++) {
            if (K[i] > 0) {
                dinic_add_edge(&din, S, i, K[i]);
                sumPos += K[i];
            } else if (K[i] < 0) {
                dinic_add_edge(&din, i, Tt, -K[i]);
            }
        }

        const ll INF = (ll)4e18;

        for (int i = 0; i < M; i++) {
            dinic_add_edge(&din, A[i], B[i], INF);
        }

        ll mincut = dinic_maxflow(&din, S, Tt);
        ll best = sumPos - mincut;

        printf("%lld\n", best > 0 ?  best : -1);
        fflush(stdout);

        dinic_free(&din);
        free(K);
        free(A);
        free(B);
    }

    return 0;
}