#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>

// Define a large number for infinite capacity (must be larger than the max possible flow)
// Max flow is sum of all positive K_i, which is at most 500 * 1024 = 512000.
#define INF 1000000000

// Maximum number of nodes: N_max + Source + Sink = 500 + 2
#define MAX_NODES 502

// Structure for an edge in the flow network
typedef struct Edge {
    int to;
    int capacity;
    int flow;
    int reverse_edge_index; // Index of the reverse edge in the 'to' node's adjacency list
} Edge;

// Adjacency list for the graph
// We use a fixed-size array of pointers to dynamically allocated arrays for simplicity in C
Edge adj[MAX_NODES][MAX_NODES];
int adj_size[MAX_NODES];

// Global variables for Dinic's algorithm
int level[MAX_NODES];
int ptr[MAX_NODES];
int num_nodes; // V = N + 2

// Function to add a directed edge and its reverse (residual) edge
void add_edge(int u, int v, int capacity) {
    // Forward edge u -> v
    adj[u][adj_size[u]].to = v;
    adj[u][adj_size[u]].capacity = capacity;
    adj[u][adj_size[u]].flow = 0;
    adj[u][adj_size[u]].reverse_edge_index = adj_size[v];
    adj_size[u]++;

    // Reverse edge v -> u (capacity 0 initially)
    adj[v][adj_size[v]].to = u;
    adj[v][adj_size[v]].capacity = 0;
    adj[v][adj_size[v]].flow = 0;
    adj[v][adj_size[v]].reverse_edge_index = adj_size[u] - 1;
    adj_size[v]++;
}

// BFS to build the level graph
int bfs(int s, int t) {
    memset(level, -1, sizeof(level));
    level[s] = 0;

    int queue[MAX_NODES];
    int head = 0, tail = 0;
    queue[tail++] = s;

    while (head < tail) {
        int u = queue[head++];
        for (int i = 0; i < adj_size[u]; i++) {
            Edge *e = &adj[u][i];
            if (e->capacity - e->flow > 0 && level[e->to] == -1) {
                level[e->to] = level[u] + 1;
                queue[tail++] = e->to;
            }
        }
    }
    return level[t] != -1;
}

// DFS to find augmenting paths in the level graph
int dfs(int u, int t, int pushed) {
    if (pushed == 0 || u == t) {
        return pushed;
    }

    for (int *i = &ptr[u]; *i < adj_size[u]; (*i)++) {
        Edge *e = &adj[u][*i];
        if (level[u] + 1 != level[e->to] || e->capacity - e->flow == 0) {
            continue;
        }

        int tr = dfs(e->to, t, (pushed < e->capacity - e->flow) ? pushed : (e->capacity - e->flow));

        if (tr == 0) {
            continue;
        }

        e->flow += tr;
        adj[e->to][e->reverse_edge_index].flow -= tr;
        return tr;
    }
    return 0;
}

// Dinic's algorithm
int dinic(int s, int t) {
    int flow = 0;
    while (bfs(s, t)) {
        memset(ptr, 0, sizeof(ptr));
        int pushed;
        while (pushed = dfs(s, t, INF)) {
            flow += pushed;
        }
    }
    return flow;
}

// Function to solve a single test case
void solve_test_case() {
    int N, M;
    if (scanf("%d %d", &N, &M) != 2) return;

    // K_i values for rooms 0 to N-1
    int K[N];
    long long total_positive_K = 0;

    for (int i = 0; i < N; i++) {
        if (scanf("%d", &K[i]) != 1) return;
        if (K[i] > 0) {
            total_positive_K += K[i];
        }
    }

    // Reset adjacency list sizes for the new graph
    memset(adj_size, 0, sizeof(adj_size));
    
    // Graph nodes: S=0, rooms 1 to N, T=N+1
    int S = 0;
    int T = N + 1;
    num_nodes = N + 2;

    // 1. Add edges from S to positive nodes and from negative nodes to T
    for (int i = 0; i < N; i++) {
        int room_node = i + 1; // Map room i (0-indexed) to node i+1 (1-indexed)
        if (K[i] > 0) {
            // S -> room_node with capacity K[i]
            add_edge(S, room_node, K[i]);
        } else if (K[i] < 0) {
            // room_node -> T with capacity |K[i]|
            add_edge(room_node, T, -K[i]);
        }
        // If K[i] == 0, no edge is added from/to S/T
    }

    // 2. Add edges for the directed tunnels (infinite capacity)
    for (int i = 0; i < M; i++) {
        int A, B;
        if (scanf("%d %d", &A, &B) != 2) return;
        
        // Map room indices (0 to N-1) to graph nodes (1 to N)
        int u = A + 1;
        int v = B + 1;
        
        // u -> v with infinite capacity
        add_edge(u, v, INF);
    }

    // 3. Calculate Max Flow (Min Cut)
    int max_flow = dinic(S, T);

    // 4. Calculate Max Weight Closure
    // Max Weight Closure = Sum of positive weights - Min Cut
    long long max_quality = total_positive_K - max_flow;

    // 5. Output the result
    if (max_quality > 0) {
        printf("%lld\n", max_quality);
    } else {
        printf("-1\n");
    }
}

// Main function to handle T test cases
int main() {
    // Optimization for faster I/O
    if (setvbuf(stdout, NULL, _IONBF, 0) != 0) {
        perror("setvbuf failed");
        return 1;
    }
    if (setvbuf(stdin, NULL, _IONBF, 0) != 0) {
        perror("setvbuf failed");
        return 1;
    }

    int T;
    if (scanf("%d", &T) != 1) return 0;

    while (T--) {
        solve_test_case();
    }

    return 0;
}